#!/usr/bin/perl -w
#
# Fix MP3 files
#
# 2005 by Patrick Stahlberg

use constant MP3CHECK_BIN => "/usr/bin/mp3check";
use constant MP3ASM_BIN => "/usr/bin/mp3asm";
use constant LAME_BIN => "/usr/local/bin/lame";
use constant MPG123_BIN => "/usr/bin/mpg123";

my $files_recoded = 0;

unless (-x MP3CHECK_BIN) {die "Error: mp3check binary not found!\n";}
#unless (-x MP3ASM_BIN) {die "Error: mp3asm binary not found!\n";}
unless (-x LAME_BIN) {die "Error: lame binary not found!\n";}
unless (-x MPG123_BIN) {die "Error: mpg123 binary not found!\n";}

my $filename;

while ($filename = shift @ARGV) {
  print "Checking `$filename'... ";

  if (mp3check($filename) == 1) {
    print "FAILED!\n";
    next;
  }

  if (mpg123($filename) == 1) {
    print "FAILED!\n";
    next;
  }

  print "OK.\n";
}

if ($files_recoded != 0) {
  print "************************************************************\n";
  print "WARNING: Some files had to be recoded.  Please check the\n" .
        "         results and delete the backup files afterwards.\n" .
        "         ( rm *.backp )\n";
  print "************************************************************\n";
}

#--------------------------------------------------------------------
# Run mp3check to test the file
sub mp3check {
  my $filename = shift;
  my $shell_filename = quote_shell($filename);
  my %fix_attempted = ();

CHECK:

  my $command = MP3CHECK_BIN . " -Be \"$shell_filename\" 2>&1";
  my @output = `$command`;

  while ($_ = shift @output) {
    chomp;
    if (/^\Q$filename\E:$/) {next;} # explained
    if (/^(?:another )?(?:in)?valid id3 tag trailer v1\.[01] found$/) {next;} # explained

    # Junk at the beginning of the file.  Remove it.
    if (/^[\d]+ bytes of junk before first frame header$/) {
      if (defined $fix_attempted{junk_start})
        {print "[again junk_start] "; return 1;}
      $fix_attempted{junk_start} = 1;
      print "[fix_junk_start] ";
      $command = MP3CHECK_BIN . " --cut-junk-start \"$shell_filename\" >/dev/null 2>&1";
      `$command`;
      goto CHECK;
    }

    # Junk at the end of the file.  Remove it.
    if (/^frame [ \d:\/]+ bytes? of junk after last frame at 0x[\da-fA-f]+$/) {
      if (defined $fix_attempted{junk_end})
        {print "[again junk_end] "; return 1;}
      $fix_attempted{junk_end} = 1;
      print "[fix_junk_end] ";
      $command = MP3CHECK_BIN . " --cut-junk-end \"$shell_filename\" >/dev/null 2>&1";
      `$command`;
      goto CHECK;
    }

    # Truncated frames:  Try to remove the last or last-but-one
    # frame of the file.  If that does not work, recode the file.
    if (/^frame +([\d]+): file truncated, [\d]+ bytes missing for last frame \(0x[\da-f]+\)$/) {
      #my $cut_frame = $1;
      #if (defined $fix_attempted{trunc_frame} and
      #    $fix_attempted{trunc_frame} == 2) {
        print "[giving up trunc_frame] ";
        next;
      #  return recode_lame($filename);
      #}
      #if (defined $fix_attempted{trunc_frame}) {
      #  $cut_frame--;
      #  $fix_attempted{trunc_frame} = 2;
      #  print "[2nd try trunc_frame] ";
      #} else {
      #  $fix_attempted{trunc_frame} = 1;
      #  print "[1st try trunc_frame] ";
      #}
      #$command = MP3ASM_BIN . " -n $cut_frame -o \"$shell_filename\" \"$shell_filename\" >/dev/null 2>&1";
      ##`$command`;                # DEBUG
      #print "CANT!!!!!! ";        # DEBUG
      #next;                       # DEBUG
      ##goto CHECK;                # DEBUG
    }

    # This is not fixable except by recoding.
    if (/^frame +([\d]+)\/ [\d:]+ frame too long at 0x[\da-f]+, skipping [\d]+ bytes at 0x[\da-f]+$/) {
        print "[frame_too_long] ";
        return recode_lame($filename);
    }

    # This is not fixable except by recoding.
    if (/^frame +([\d]+): invalid header at 0x[\da-f]+ \(0x[\da-f]+\), skipping [\d]+ bytes$/) {
        print "[invalid_header] ";
        return recode_lame($filename);
    }

    # This is not fixable except by recoding.
    if (/^frame +([\d]+)\/ [\d:]+ frame too short at 0x[\da-f]+, [\d]+ bytes mising$/) {
        print "[frame_too_long] ";
        return recode_lame($filename);
    }

    # This is not fixable except by recoding.
    if (/^frame +([\d]+): constant parameter switching \(0x[\da-f]+!=0x[\da-f]+\)$/) {
        print "[constant_parameter_switch] ";
        return recode_lame($filename);
    }

    # This is not fixable except by recoding.
    if (/^frame +([\d]+)\/ [\d:]+ crc error \(0x[\da-f]+!=0x[\da-f]+\)$/) {
        print "[crc_error] ";
        return recode_lame($filename);
    }

    print "unexplained mp3check output: '$_' ";
    return 1;
  }

  return 0;
}

#--------------------------------------------------------------------
# Run mpg123 -t to test the file
sub mpg123 {
  my $filename = shift;
  my $shell_filename = quote_shell($filename);

  my $command = MPG123_BIN . " -t \"$shell_filename\" 2>&1";
  my @output = `$command`;

  while ($_ = shift @output) {
    chomp;
    if (/^$/) {next;} # explained
    if (/^High Performance MPEG 1\.0\/2\.0\/2\.5 Audio Player for Layer 1, 2,? and 3\.$/) {next;} # explained
    if (/^Version 0\.59r \(1999\/Jun\/15\)\. Written and copyrights by Michael Hipp\.$/) {next;} # explained
    if (/^Version 0\.59s-mh4 \(2000\/Oct\/27\)\. Written and copyrights by Michael Hipp\.$/) {next;} # explained
    if (/^Version 0\.59q \(2002\/03\/23\)\. Written and copyrights by Joe Drew\.$/) {next;} # explained
    if (/^Found XING 000f$/) {next;} # explained
    if (/^Found old ID3 Header$/) {next;} # explained
    if (/^Uses code from various people\. See 'README' for more!$/) {next;} # explained
    if (/^THIS SOFTWARE COMES WITH ABSOLUTELY NO WARRANTY! USE AT YOUR OWN RISK!$/) {next;} # explained
    if (/^Title  : .*Artist: .*$/) {next;} # explained
    if (/^Album  : .*Year  : .*$/) {next;} # explained
    if (/^Comment: .*Genre : .*$/) {next;} # explained
    if (/^Playing MPEG stream from \Q$filename\E ...$/) {next;} # explained
    if (/^MPEG [12]\.0 layer III?, ([\d]+) kbit\/s, [\d]+ Hz (?:joint-)?stereo$/) {next;} # explained
    if (/^MPEG [12]\.0 layer III?, ([\d]+) kbit\/s, [\d]+ Hz mono$/) {next;} # explained
    if (/^\[[\d:]+\] Decoding of \Q$filename\E finished.$/) {next;} # explained
    if (/^Illegal Audio-MPEG-Header 0x[\dabcdef]+ at offset 0x[\dabcdef]+\.$/) {next;} # explained...sort of

    # some serious error in the mp3 file. Recoding needed.
    if (/^mpg123: Can't rewind stream by [\d]+ bits!$/) {
      print "[cant_rewind_error] ";
      return recode_lame($filename);
    }

    # some serious error in the mp3 file. Recoding needed.
    if (/^big_values too large!$/) {
      print "[big_values_error] ";
      return recode_lame($filename);
    }

    # some serious error in the mp3 file. Recoding needed.
    if (/^Blocktype == 0 and window-switching == 1 not allowed\.$/) {
      print "[blocktype_error] ";
      return recode_lame($filename);
    }

    print "unexplained mpg123 output: '$_' ";
    return 1;
  }

  return 0;
}

#--------------------------------------------------------------------
# Use lame to recode the mp3 file.
sub recode_lame {
  my $filename = shift;
  my $shell_filename = quote_shell($filename);
  my $bitrate;
  my $command;

  # find out the bitrate
  $command = MPG123_BIN . " -t -n 1 \"$shell_filename\" 2>&1 " .
                                          "| grep \"MPEG 1.0 layer\"";
  if (`$command` =~ / ([\d]+) kbit\/s/) {
    $bitrate = $1;
  } else {
    print "[recode: can't determine bitrate] ";
    return 1;
  }
  $command = MPGCHECK_BIN . " -e \"$shell_filename\" 2>&1 " .
                                          "| grep -c \"bitrate switching\"";
  if (`$command` >= 1) {
    $bitrate = 160;
  }

  # make a backup
  $command = "mv \"$shell_filename\" \"$shell_filename.backup\"";
  `$command`;
  unless ( -f "$shell_filename.backup" ) {
    print "[recode: can't make backup] ";
    return 1;
  }

  # do the conversion
  print "[recoding with $bitrate kbit/s...";
  $command = LAME_BIN . " -h -b $bitrate --mp3input " .
                                     "\"$shell_filename.backup\" ".
                                     "\"$shell_filename\" >/dev/null 2>&1";
  `$command`;
  print " ok] ";

  # store the filename, for the warning at the end
  $files_recoded = 1;

  return 0;
}

#--------------------------------------------------------------------
# Quote a string so that it can be given to a shell
sub quote_shell {
  $_ = shift;
  $_ =~ s/\\/\\\\/g;
  $_ =~ s/"/\\"/g;
  $_ =~ s/\$/\\\$/g;
  return $_
}
